/*************************************************************************************************
 * The CGI search utility of the core API
 *                                                      Copyright (C) 2007-2008 Mikio Hirabayashi
 * This file is part of Tokyo Dystopia.
 * Tokyo Dystopia is free software; you can redistribute it and/or modify it under the terms of
 * the GNU Lesser General Public License as published by the Free Software Foundation; either
 * version 2.1 of the License or any later version.  Tokyo Dystopia is distributed in the hope
 * that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU Lesser General Public
 * License for more details.
 * You should have received a copy of the GNU Lesser General Public License along with Tokyo
 * Dystopia; if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307 USA.
 *************************************************************************************************/


#include <dystopia.h>
#include "myconf.h"

#define INDEXNAME      "casket"          // name of the index
#define MINIBNUM       31                // bucket number of map for trivial use
#define OUTBUFSIZ      (256*1024)        // size of the output buffer
#define UPLOADMAX      (256*1024*1024)   // maximum size of the upload data
#define DEFSHOWNUM     30                // default number of show result
#define PAGETITLE      "Full-text Search by Tokyo Dystopia"  // page title

#define XP(...) tcxstrprintf(obuf, __VA_ARGS__)  // syntex sugar of output setting

typedef struct {                         // type of structure of CGI parameters
  const char *phrase;                    // search phrase
  int emode;                             // expresion mode
  int smode;                             // search mode
  int num;                               // number per page
  int page;                              // number of the page
} PARAMS;

enum {                                   // enumeration for expression modes
  EMCOMP,                                // compound
  EMPHRASE,                              // phrase
  EMUNION,                               // union
  EMISECT,                               // intersection
  EMDIFF,                                // difference
};


/* global variables */
const char *g_scriptname;                // program name


/* function prototypes */
int main(int argc, char **argv);
static void readparameters(TCMAP *map);
static void writeheader(const PARAMS *params, TCXSTR *obuf);
static void writeresult(const PARAMS *params, TCXSTR *obuf);
static void writefooter(const PARAMS *params, TCXSTR *obuf);


/* main routine */
int main(int argc, char **argv){
  g_scriptname = getenv("SCRIPT_NAME");
  if(!g_scriptname) g_scriptname = argv[0];
  const char *rp = strrchr(g_scriptname, '/');
  if(rp) g_scriptname = rp + 1;
  TCMAP *pmap = tcmapnew2(MINIBNUM);
  readparameters(pmap);
  PARAMS params;
  const char *str;
  params.phrase = (str = tcmapget2(pmap, "phrase")) ? str : "";
  params.emode = (str = tcmapget2(pmap, "emode")) ? tclmax(atoi(str), 0) : 0;
  params.smode = (str = tcmapget2(pmap, "smode")) ? tclmax(atoi(str), 0) : 0;
  params.num = (str = tcmapget2(pmap, "num")) ? tclmax(atoi(str), 0) : 0;
  params.page = (str = tcmapget2(pmap, "page")) ? tclmax(atoi(str), 0) : 0;
  if(params.num < 1) params.num = DEFSHOWNUM;
  if(params.page < 1) params.page = 1;
  TCXSTR *obuf = tcxstrnew3(OUTBUFSIZ);
  printf("Cache-Control: no-cache, must-revalidate, no-transform\r\n");
  printf("Pragma: no-cache\r\n");
  printf("Content-Disposition: inline; filename=%s\r\n", g_scriptname);
  printf("Content-Type: text/html; charset=UTF-8\r\n");
  printf("\r\n");
  writeheader(&params, obuf);
  if(*params.phrase != '\0') writeresult(&params, obuf);
  writefooter(&params, obuf);
  fwrite(tcxstrptr(obuf), 1, tcxstrsize(obuf), stdout);
  tcxstrdel(obuf);
  tcmapdel(pmap);
  return 0;
}


/* read CGI parameters */
static void readparameters(TCMAP *map){
  int maxlen = UPLOADMAX;
  char *buf = NULL;
  int len = 0;
  char *rp;
  if((rp = getenv("REQUEST_METHOD")) != NULL && !strcmp(rp, "POST") &&
     (rp = getenv("CONTENT_LENGTH")) != NULL && (len = atoi(rp)) > 0){
    if(len > maxlen) len = maxlen;
    buf = tccalloc(len + 1, 1);
    if(fread(buf, 1, len, stdin) != len){
      tcfree(buf);
      buf = NULL;
    }
  } else if((rp = getenv("QUERY_STRING")) != NULL){
    buf = tcstrdup(rp);
    len = strlen(buf);
  }
  if(buf && len > 0){
    TCLIST *pairs = tcstrsplit(buf, "&");
    int num = tclistnum(pairs);
    for(int i = 0; i < num; i++){
      char *key = tcstrdup(tclistval2(pairs, i));
      char *val = strchr(key, '=');
      if(val){
        *(val++) = '\0';
        char *dkey = tcurldecode(key, &len);
        char *dval = tcurldecode(val, &len);
        tcmapput2(map, dkey, dval);
        tcfree(dval);
        tcfree(dkey);
      }
      tcfree(key);
    }
    tclistdel(pairs);
  }
  tcfree(buf);
}


/* write the herder section */
static void writeheader(const PARAMS *params, TCXSTR *obuf){
  XP("<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n");
  XP("\n");
  XP("<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.0 Strict//EN\""
     " \"http://www.w3.org/TR/xhtml1/DTD/xhtml1-strict.dtd\">\n");
  XP("\n");
  XP("<html xmlns=\"http://www.w3.org/1999/xhtml\" xml:lang=\"en\" lang=\"en\">\n");
  XP("\n");
  XP("<head>\n");
  XP("<meta http-equiv=\"Content-Type\" content=\"text/html; charset=UTF-8\" />\n");
  XP("<meta http-equiv=\"Content-Style-Type\" content=\"text/css\" />\n");
  XP("<meta http-equiv=\"Content-Script-Type\" content=\"text/javascript\" />\n");
  XP("<title>%@</title>\n", PAGETITLE);
  XP("<style type=\"text/css\">\n");
  XP("html { margin: 0em; padding: 0em; }\n");
  XP("body { margin :0em; padding: 0.5em 1em; background: #eeeeee none; color: #111111; }\n");
  XP("hr { margin: 2.0em 0em 1.0em 0em; height: 1px;"
     " border: none; background: #999999 none; color: #999999; }\n");
  XP("h1, h2, h3, h4, h5, h6 { font-size: large; }\n");
  XP("a { color: #2277ee; text-decoration: none; }\n");
  XP("div.form { margin: 5px 8px; }\n");
  XP("div.advancedform option, div.advancedform input { font-size: 95%%; }\n");
  XP("#result { margin: 10px 5px; padding: 5px 5px; }\n");
  XP("p { margin: 10px 10px; font-size: 95%%; color: #555555; }\n");
  XP("#watch { text-align: right; float: right; font-size: 90%%; color: #888888; }\n");
  XP("#navi { font-size: 90%%; color: #888888; }\n");
  XP("span.id { font-size: 95%%; color: #555555; }\n");
  XP("span.delim { font-size: 95%%; color: #888888; }\n");
  XP("div.footer { text-align: right; font-size: 90%%; color: #888888; }\n");
  XP("</style>\n");
  XP("<script type=\"text/javascript\">\n");
  XP("function validateform(){\n");
  XP("  var inputemode = document.getElementById(\"inputemode\");\n");
  XP("  var inputsmode = document.getElementById(\"inputsmode\");\n");
  XP("  if(inputemode.value == 0){\n");
  XP("    inputsmode.disabled = true;\n");
  XP("  } else {\n");
  XP("    inputsmode.disabled = false;\n");
  XP("  }\n");
  XP("}\n");
  XP("</script>\n");
  XP("</head>\n");
  XP("\n");
  XP("<body onload=\"validateform();\">\n");
  XP("\n");
  XP("<form method=\"get\" action=\"%@\">\n", g_scriptname);
  XP("<div class=\"form basicform\">\n");
  XP("<input type=\"text\" name=\"phrase\" value=\"%@\" id=\"inputphrase\" size=\"64\" />\n",
     params->phrase);
  XP("<input type=\"submit\" value=\"Search\" id=\"submitsearch\" />\n");
  XP("</div>\n");
  XP("<div class=\"form advancedform\">\n");
  XP("<select name=\"emode\" id=\"inputemode\" onchange=\"validateform();\">\n");
  XP("<option value=\"%d\"%s>compound expression</option>\n",
     EMCOMP, (params->emode == EMCOMP) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>phrase expression</option>\n",
     EMPHRASE, (params->emode == EMPHRASE) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>union expression</option>\n",
     EMUNION, (params->emode == EMUNION) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>intersection expression</option>\n",
     EMISECT, (params->emode == EMISECT) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>difference expression</option>\n",
     EMDIFF, (params->emode == EMDIFF) ? " selected=\"selected\"" : "");
  XP("</select>\n");
  XP("<select name=\"smode\" id=\"inputsmode\" onchange=\"validateform();\">\n");
  XP("<option value=\"%d\"%s>substring matching</option>\n",
     IDBSSUBSTR, (params->smode == IDBSSUBSTR) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>prefix matching</option>\n",
     IDBSPREFIX, (params->smode == IDBSPREFIX) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>suffix matching</option>\n",
     IDBSSUFFIX, (params->smode == IDBSSUFFIX) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>full matching</option>\n",
     IDBSFULL, (params->smode == IDBSFULL) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>token matching</option>\n",
     IDBSTOKEN, (params->smode == IDBSTOKEN) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>token prefix matching</option>\n",
     IDBSTOKPRE, (params->smode == IDBSTOKPRE) ? " selected=\"selected\"" : "");
  XP("<option value=\"%d\"%s>token suffix matching</option>\n",
     IDBSTOKSUF, (params->smode == IDBSTOKSUF) ? " selected=\"selected\"" : "");
  XP("</select>\n");
  XP("<select name=\"num\" id=\"inputenum\" onchange=\"validateform();\">\n");
  for(int i = 10; i <= 100; i += 10){
    XP("<option value=\"%d\"%s>%d per page</option>\n",
       i, (i == params->num) ? " selected=\"selected\"" : "", i);
  }
  XP("</select>\n");
  XP("</div>\n");
  XP("</form>\n");
  XP("\n");
}


/* write the result section */
static void writeresult(const PARAMS *params, TCXSTR *obuf){
  XP("<hr />\n");
  XP("\n");
  XP("<div id=\"result\">\n");
  TCIDB *idb = tcidbnew();
  if(tcidbopen(idb, INDEXNAME, IDBOREADER | IDBONOLCK)){
    uint64_t *res = NULL;
    int rnum = 0;
    double stime = tctime();
    if(params->emode == EMCOMP){
      res = tcidbsearch2(idb, params->phrase, &rnum);
    } else if(params->emode == EMPHRASE || !strchr(params->phrase, ' ')){
      res = tcidbsearch(idb, params->phrase, params->smode, &rnum);
    } else if(params->emode == EMUNION){
      char *tphrase = tcstrdup(params->phrase);
      tctextnormalize(tphrase, TCTNSPACE);
      TCLIST *words = tcstrsplit(tphrase, " ");
      int wnum = tclistnum(words);
      QDBRSET rsets[wnum];
      for(int i = 0; i < wnum; i++){
        rsets[i].ids = tcidbsearch(idb, tclistval2(words, i), params->smode, &rsets[i].num);
      }
      res = tcqdbresunion(rsets, wnum, &rnum);
      for(int i = 0; i < wnum; i++){
        tcfree(rsets[i].ids);
      }
      tclistdel(words);
      tcfree(tphrase);
    } else if(params->emode == EMISECT){
      char *tphrase = tcstrdup(params->phrase);
      tctextnormalize(tphrase, TCTNSPACE);
      TCLIST *words = tcstrsplit(tphrase, " ");
      int wnum = tclistnum(words);
      QDBRSET rsets[wnum];
      for(int i = 0; i < wnum; i++){
        rsets[i].ids = tcidbsearch(idb, tclistval2(words, i), params->smode, &rsets[i].num);
      }
      res = tcqdbresisect(rsets, wnum, &rnum);
      for(int i = 0; i < wnum; i++){
        tcfree(rsets[i].ids);
      }
      tclistdel(words);
      tcfree(tphrase);
    } else if(params->emode == EMDIFF){
      char *tphrase = tcstrdup(params->phrase);
      tctextnormalize(tphrase, TCTNSPACE);
      TCLIST *words = tcstrsplit(tphrase, " ");
      int wnum = tclistnum(words);
      QDBRSET rsets[wnum];
      for(int i = 0; i < wnum; i++){
        rsets[i].ids = tcidbsearch(idb, tclistval2(words, i), params->smode, &rsets[i].num);
      }
      res = tcqdbresdiff(rsets, wnum, &rnum);
      for(int i = 0; i < wnum; i++){
        tcfree(rsets[i].ids);
      }
      tclistdel(words);
      tcfree(tphrase);
    }
    XP("<div id=\"watch\">elapsed time: %.6f</div>\n", tctime() - stime);
    if(res){
      if(rnum > 0){
        char *baseurl = tcsprintf("%s?phrase=%?&emode=%d&smode=%d&num=%d", g_scriptname,
                                  params->phrase, params->emode, params->smode, params->num);
        int start = (params->page - 1) * params->num;
        int end = tclmin((params->page) * params->num, rnum);
        int fpage = tclmax(1, params->page - 9);
        int lpage = tclmin(rnum / params->num + 1, params->page + 9);
        XP("<div id=\"navi\">\n");
        if(rnum > 1){
          XP("<span class=\"page\">%d records hit:</span>\n", rnum);
        } else {
          XP("<span class=\"page\">%d record hits:</span>", rnum);
        }
        if(params->page > fpage){
          XP("<a href=\"%@&amp;page=%d\" class=\"page jump\">&lt;&lt;</a>\n",
             baseurl, params->page - 1);
        } else {
          XP("<span class=\"page void\">&lt;&lt;</span>\n");
        }
        for(int i = fpage; i < params->page; i++){
          XP("<a href=\"%@&amp;page=%d\" class=\"page jump\">[%d]</a>\n", baseurl, i, i);
        }
        XP("<span class=\"page void\">[%d]</span>\n", params->page);
        for(int i = params->page + 1; i <= lpage; i++){
          XP("<a href=\"%@&amp;page=%d\" class=\"page jump\">[%d]</a>\n", baseurl, i, i);
        }
        if(params->page < lpage){
          XP("<a href=\"%@&amp;page=%d\" class=\"page jump\">&gt;&gt;</a>\n",
             baseurl, params->page + 1);
        } else {
          XP("<span class=\"page void\">&gt;&gt;</span>\n");
        }
        XP("</div>\n");
        tcfree(baseurl);
        XP("<ul id=\"resultlist\">\n");
        for(int i = start; i < end; i++){
          uint64_t id = res[i];
          char *text = tcidbget(idb, id);
          if(text){
            XP("<li><span class=\"id\">%llu:</span> <span class=\"word\">", id);
            for(int j = 0; text[j] != '\0'; j++){
              switch(text[j]){
              case '<': XP("&lt;"); break;
              case '>': XP("&gt;"); break;
              case '&': XP("&amp;"); break;
              case '\t': XP(" <span class=\"delim\">-</span> "); break;
              default: tcxstrcat(obuf, text + j, 1); break;
              }
            }
            XP("</span></li>\n", id);
            tcfree(text);
          } else {
            XP("<li><span class=\"id\">%llu:</span> (error)</li>\n", id);
          }
        }
        XP("</ul>\n");
      } else {
        XP("<p>No record hits.</p>\n", rnum);
      }
      tcfree(res);
    } else {
      XP("<p>Search operation failed.</p>\n");
    }
  } else {
    XP("<p>The index could not be opened.</p>\n");
  }
  tcidbdel(idb);
  XP("</div>\n");
  XP("\n");
}


/* write the footer section */
static void writefooter(const PARAMS *params, TCXSTR *obuf){
  XP("<hr />\n");
  XP("\n");
  XP("<div class=\"footer\">\n");
  XP("powered by Tokyo Dystopia version %s.\n", tdversion);
  XP("</div>\n");
  XP("\n");
  XP("</body>\n");
  XP("\n");
  XP("</html>\n");
}



// END OF FILE
