/*************************************************************************************************
 * Command line utility to view documents in index
 *                                                      Copyright (C) 2003-2006 Mikio Hirabayashi
 * This file is part of Estraier, a personal full-text search system.
 * Estraier is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * Estraier is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Estraier;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "estcommon.h"

#define CSSFILE     "estxview.css"       /* name of the CSS file */
#define XSLFILE     "estxview.xsl"       /* name of the XSL file */
#define DTDFILE     "estxview.dtd"       /* name of the DTD file */
#define ENCODING    "UTF-8"              /* default character encoding */
#define DEFMAX      8                    /* number of shown documents by default */
#define RELKEYS     16                   /* number of words used with relational search */
#define RELVECNUM   32                   /* number of dimension of score vector */
#define RESROOM     8                    /* ratio of hits for max */
#define UNITINIT    1024                 /* unit number of boolean search */
#define UNITINC     4                    /* inclement coefficient of boolean unit */
#define REEVMAX     1024                 /* max number of candidates of a regular expression */
#define DEFSNUM     80                   /* number of words in summary by default */
#define SUMTOP      16                   /* number of words at the top of summary */
#define SUMWIDTH    16                   /* number of words near each keyword in summary */
#define CLUSTUNIT   128                  /* unit number of document clustering */
#define CLUSTKEYS   32                    /* number of words per shown cluster */

typedef struct {                         /* type of structure for a cluster */
  CBLIST *ids;                           /* list for ID numbers */
  CBMAP *scores;                         /* map for scores */
  int vector[RELVECNUM];                 /* vector for calculating similarity */
  int point;                             /* total point */
} CLUSTER;

typedef struct {                         /* type of structure to sort scores */
  const char *kbuf;                      /* pointer to the string */
  int ksiz;                              /* size of the string */
  int point;                             /* total point */
} SCWORD;


/* global variables */
const char *progname = NULL;             /* program name */
ODEUM *odeum = NULL;                     /* handle of the index */
CURIA *scdb = NULL;                      /* handle of the score database */
DEPOT *dtdb = NULL;                      /* handle of the date database */
int optmax = DEFMAX;                     /* default number of shown documents */
int optclshow = 0;                       /* number of shown clusters */
int optclcode = 0;                       /* code of the cluster for narrowing */
int optdrep = 0;                         /* level of directory rep */
const char *optetype = "asis";           /* expression format of search words */
const char *optstype = "score";          /* order of sorting the result */
int optni = FALSE;                       /* whether TF-IDF to be disabled */
int opttiny = FALSE;                     /* whether to show documents in tiny format */
int optnt = FALSE;                       /* whether to hide text of documents */
int optsnum = DEFSNUM;                   /* number of words in summary */
int optnk = FALSE;                       /* whether to hide keywords of documents */
int optcss = FALSE;                      /* whether to show the process instruction for CSS */
int optxsl = FALSE;                      /* whether to show the process instruction for XSLT */
int optdtd = FALSE;                      /* whether to show the DTD reference */
const char *opticode = NULL;             /* character encoding of the input */
const char *optocode = NULL;             /* character encoding of the output */


/* function prototypes */
int main(int argc, char **argv);
void usage(void);
int dosearch(const char *name, const char *expr, int ib, int ub, int rel, int relsc);
void xmlprintf(const char *format, ...);
void showerror(const char *msg, int code);
const char *getoptionstr(void);
void setovec(CBMAP *scores, int *vec);
void settvec(CBMAP *osc, CBMAP *tsc, int *vec);
void showclusters(ODPAIR *pairs, int *np);
void sumupclusters(CLUSTER *clusts, int *np);
int scwordcompare(const void *a, const void *b);
int clustercompare(const void *a, const void *b);
char *cuturi(const char *uri, int level);
void showpreamble(void);
void showdoc(const ODDOC *doc, int score, double rel, ESTWORD *words, int wnum);
void showsummary(const ODDOC *doc, const ESTWORD *words, int wnum);
void showdoctiny(int id, int score, double rel);
int showdocbyid(int id);
int showdocbyuri(const char *uri);
int showdocbyrel(int id, const char *relsc);
int showdocbywords(ESTWORD *words, int wnum);
int dodlist(const char *name);
int dowlist(const char *name);


/* main routine */
int main(int argc, char **argv){
  CBDATUM *expr;
  char *name, *tmp;
  int i, ib, ub, rel, relsc, rf, dlist, wlist;
  estputenv("LANG", ESTLOCALE);
  estputenv("LC_ALL", ESTLOCALE);
  if((tmp = getenv(ESTDBGFDENV)) != NULL) dpdbgfd = atoi(tmp);
  cbstdiobin();
  progname = argv[0];
  name = NULL;
  expr = cbdatumopen("", 0);
  cbglobalgc(expr, (void (*)(void *))cbdatumclose);
  ib = FALSE;
  ub = FALSE;
  rel = FALSE;
  relsc = FALSE;
  rf = FALSE;
  dlist = FALSE;
  wlist = FALSE;
  for(i = 1; i < argc; i++){
    if(!name && argv[i][0] == '-'){
      if(!strcmp(argv[i], "-id")){
        ib = TRUE;
      } else if(!strcmp(argv[i], "-uri")){
        ub = TRUE;
      } else if(!strcmp(argv[i], "-rel")){
        rel = TRUE;
      } else if(!strcmp(argv[i], "-relsc")){
        relsc = TRUE;
      } else if(!strcmp(argv[i], "-expr")){
        if(++i >= argc) usage();
        optetype = argv[i];
      } else if(!strcmp(argv[i], "-sort")){
        if(++i >= argc) usage();
        optstype = argv[i];
      } else if(!strcmp(argv[i], "-max")){
        if(++i >= argc) usage();
        optmax = atoi(argv[i]);
      } else if(!strcmp(argv[i], "-drep")){
        if(++i >= argc) usage();
        optdrep = atoi(argv[i]);
      } else if(!strcmp(argv[i], "-clshow")){
        if(++i >= argc) usage();
        optclshow = atoi(argv[i]);
      } else if(!strcmp(argv[i], "-clcode")){
        if(++i >= argc) usage();
        optclcode = atoi(argv[i]);
      } else if(!strcmp(argv[i], "-ni")){
        optni = TRUE;
      } else if(!strcmp(argv[i], "-tiny")){
        opttiny = TRUE;
      } else if(!strcmp(argv[i], "-nt")){
        optnt = TRUE;
      } else if(!strcmp(argv[i], "-snum")){
        if(++i >= argc) usage();
        optsnum = atoi(argv[i]);
      } else if(!strcmp(argv[i], "-nk")){
        optnk = TRUE;
      } else if(!strcmp(argv[i], "-css")){
        optcss = TRUE;
      } else if(!strcmp(argv[i], "-xsl")){
        optxsl = TRUE;
      } else if(!strcmp(argv[i], "-dtd")){
        optdtd = TRUE;
      } else if(!strcmp(argv[i], "-rf")){
        rf = TRUE;
      } else if(!strcmp(argv[i], "-dlist")){
        dlist = TRUE;
      } else if(!strcmp(argv[i], "-wlist")){
        wlist = TRUE;
      } else if(!strcmp(argv[i], "-ic")){
        if(++i >= argc) usage();
        opticode = argv[i];
      } else if(!strcmp(argv[i], "-oc")){
        if(++i >= argc) usage();
        optocode = argv[i];
      } else {
        usage();
      }
    } else if(!name){
      name = argv[i];
    } else {
      if(cbdatumsize(expr) > 0) cbdatumcat(expr, " ", -1);
      cbdatumcat(expr, argv[i], -1);
    }
  }
  if(!name) usage();
  if(strcmp(optetype, "asis") && strcmp(optetype, "wild") && strcmp(optetype, "regex")) usage();
  if(strcmp(optstype, "score") && strcmp(optstype, "r-score") &&
     strcmp(optstype, "date") && strcmp(optstype, "r-date")) usage();
  if(optmax < 0) usage();
  if(optclshow < 0) usage();
  if(optclcode < 0) usage();
  if(optdrep < 0) usage();
  if(rf && cbdatumsize(expr) > 0){
    if(!(tmp = cbreadfile(cbdatumptr(expr), NULL))){
      fprintf(stderr, "%s: %s: can not open\n", progname, cbdatumptr(expr));
      exit(1);
    }
    cbdatumsetsize(expr, 0);
    cbdatumcat(expr, tmp, -1);
    free(tmp);
  }
  if(opticode){
    if(!(tmp = cbiconv(cbdatumptr(expr), cbdatumsize(expr), opticode, ENCODING, NULL, NULL))){
      fprintf(stderr, "%s: %s: invalid encoding\n", progname, opticode);
      exit(1);
    }
    cbdatumsetsize(expr, 0);
    cbdatumcat(expr, tmp, -1);
    free(tmp);
  }
  if(optocode){
    if(!(tmp = cbiconv("dummy", -1, optocode, ENCODING, NULL, NULL))){
      fprintf(stderr, "%s: %s: invalid encoding\n", progname, optocode);
      exit(1);
    }
    free(tmp);
  }
  if(cbstrfwimatch(cbdatumptr(expr), "[related] ")){
    rel = TRUE;
    tmp = cbmemdup(strchr(cbdatumptr(expr), ' ') + 1, -1);
    cbdatumsetsize(expr, 0);
    cbdatumcat(expr, tmp, -1);
    free(tmp);
  } else if(cbstrfwimatch(cbdatumptr(expr), "[detail] ")){
    ib = TRUE;
    tmp = cbmemdup(strchr(cbdatumptr(expr), ' ') + 1, -1);
    cbdatumsetsize(expr, 0);
    cbdatumcat(expr, tmp, -1);
    free(tmp);
  }
  if(dlist){
    return dodlist(name);
  } else if(wlist){
    return dowlist(name);
  }
  return dosearch(name, cbdatumptr(expr), ib, ub, rel, relsc);
}


/* print the usage and exit */
void usage(void){
  fprintf(stderr, "%s: viewer documents in index\n", progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s [-id] [-uri] [-rel] [-relsc] [-expr type] [-sort type] [-drep num]"
          " [-clshow num] [-clcode num] [-max num] [-ni] [-tiny] [-nt] [-snum num] [-nk]"
          " [-css] [-xsl] [-dtd] [-rf] [-dlist] [-wlist] [-ic] [-oc] name [expression...]\n",
          progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "typical examples:\n");
  fprintf(stderr, "  %s casket \"apple orange\"  # search for documents including `apple' and"
          " `orange'\n", progname);
  fprintf(stderr, "  %s -id casket 1           # search for the document whose ID is 1\n",
          progname);
  fprintf(stderr, "  %s -uri casket ./foo.txt  # search for the document whose URI is"
          " `./foo.txt'\n", progname);
  fprintf(stderr, "  %s -rel casket 1          # search for documents related to the document"
          " whose ID is 1\n", progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* search documents */
int dosearch(const char *name, const char *expr, int ib, int ub, int rel, int relsc){
  ESTWORD *words;
  char scdbpath[ESTPATHBUFSIZ], dtdbpath[ESTPATHBUFSIZ];
  int err, id, wnum;
  err = FALSE;
  if(!(odeum = odopen(name, OD_OREADER))){
    fprintf(stderr, "%s: %s: could not open: %s\n", progname, name, dperrmsg(dpecode));
    err = TRUE;
  } else {
    sprintf(scdbpath, "%s%c%s", name, ESTPATHCHR, ESTSCDBNAME);
    scdb = cropen(scdbpath, CR_OREADER, -1, -1);
    sprintf(dtdbpath, "%s%c%s", name, ESTPATHCHR, ESTDTDBNAME);
    dtdb = dpopen(dtdbpath, DP_OREADER, -1);
    if(ib){
      if((id = atoi(expr)) < 1){
        fprintf(stderr, "%s: %s: invalid ID expression\n", progname, expr);
        err = TRUE;
      } else {
        if(!showdocbyid(id)) err = TRUE;
      }
    } else if(ub){
      if(strlen(expr) < 1){
        fprintf(stderr, "%s: %s: invalid URI expression\n", progname, expr);
        err = TRUE;
      } else {
        if(!showdocbyuri(expr)) err = TRUE;
      }
    } else if(rel || relsc){
      if(!scdb){
        fprintf(stderr, "%s: %s: could not open\n", progname, scdbpath);
        err = TRUE;
      } else if(relsc){
        if(!showdocbyrel(-1, expr)) err = TRUE;
      } else {
        if((id = atoi(expr)) < 1){
          fprintf(stderr, "%s: %s: invalid ID expression\n", progname, expr);
          err = TRUE;
        } else {
          if(!showdocbyrel(id, NULL)) err = TRUE;
        }
      }
    } else {
      words = estsearchwords(expr, &wnum, !strcmp(optetype, "asis"));
      if(wnum < 1){
        fprintf(stderr, "%s: %s: no effective word in the phrase\n", progname, expr);
        err = TRUE;
      } else {
        if(!showdocbywords(words, wnum)) err = TRUE;
      }
      estfreewords(words, wnum);
    }
    if(dtdb) dpclose(dtdb);
    if(scdb) crclose(scdb);
  }
  if(odeum) odclose(odeum);
  return err ? 1 : 0;
}


/* XML-oriented printf */
void xmlprintf(const char *format, ...){
  va_list ap;
  char *tmp, *enc;
  unsigned char c;
  va_start(ap, format);
  while(*format != '\0'){
    if(*format == '%'){
      format++;
      switch(*format){
      case 's':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        printf("%s", tmp);
        break;
      case 'd':
        printf("%d", va_arg(ap, int));
        break;
      case '@':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        enc = NULL;
        if(optocode && (enc = cbiconv(tmp, -1, ENCODING, optocode, NULL, NULL)) != NULL)
          tmp = enc;
        while(*tmp){
          switch(*tmp){
          case '&': printf("&amp;"); break;
          case '<': printf("&lt;"); break;
          case '>': printf("&gt;"); break;
          case '"': printf("&quot;"); break;
          default:
            if(!((*tmp >= 0 && *tmp <= 0x8) || (*tmp >= 0x0e && *tmp <= 0x1f))) putchar(*tmp);
            break;
          }
          tmp++;
        }
        if(enc) free(enc);
        break;
      case '?':
        tmp = va_arg(ap, char *);
        if(!tmp) tmp = "(null)";
        while(*tmp){
          c = *(unsigned char *)tmp;
          if((c >= 'A' && c <= 'Z') || (c >= 'a' && c <= 'z') ||
             (c >= '0' && c <= '9') || (c != '\0' && strchr("_-.", c))){
            putchar(c);
          } else {
            printf("%%%02X", c);
          }
          tmp++;
        }
        break;
      case '%':
        putchar('%');
        break;
      }
    } else {
      putchar(*format);
    }
    format++;
  }
  va_end(ap);
}


/* show an error message */
void showerror(const char *msg, int code){
  if(code >= 0){
    xmlprintf("<error code=\"%d\" string=\"%@\">%@</error>\n", code, dperrmsg(dpecode), msg);
  } else {
    xmlprintf("<error>%@</error>\n", msg);
  }
}


/* get the string of the building options */
const char *getoptionstr(void){
  static char str[ESTPATHBUFSIZ], *wp;
  wp = str;
  if(estisregex){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "regex");
  }
  if(ESTISSTRICT){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "strict");
  }
  if(ESTISNOSTOPW){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "no-stopword");
  }
  if(estiscjkuni){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "cjkuni");
  }
  if(estischasen){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "chasen");
  }
  if(estismecab){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "mecab");
  }
  if(estiskakasi){
    if(wp != str) *(wp++) = ' ';
    wp += sprintf(wp, "kakasi");
  }
  *wp = '\0';
  return str;
}


/* set the original score vector */
void setovec(CBMAP *scores, int *vec){
  const char *kbuf;
  int i, ksiz;
  cbmapiterinit(scores);
  for(i = 0; i < RELVECNUM; i++){
    if((kbuf = cbmapiternext(scores, &ksiz)) != NULL){
      vec[i] = atoi(cbmapget(scores, kbuf, ksiz, NULL));
    } else {
      vec[i] = 0;
    }
  }
}


/* set the target score vector */
void settvec(CBMAP *osc, CBMAP *tsc, int *vec){
  const char *kbuf, *vbuf;
  int i, ksiz;
  cbmapiterinit(osc);
  for(i = 0; i < RELVECNUM; i++){
    if((kbuf = cbmapiternext(osc, &ksiz)) != NULL){
      vbuf = cbmapget(tsc, kbuf, ksiz, NULL);
      vec[i] = vbuf ? atoi(vbuf) : 0;
    } else {
      vec[i] = 0;
    }
  }
}


/* show clusters */
void showclusters(ODPAIR *pairs, int *np){
  CLUSTER *clusts;
  CBMAP **tails;
  ODPAIR *tpairs;
  const char *kbuf;
  char *mbuf, numbuf[ESTNUMBUFSIZ];
  int i, j, pnum, blnum, clnum, tnum, msiz, mi, ovec[RELVECNUM], tvec[RELVECNUM];
  double msc, sc;
  pnum = *np;
  if(pnum > CLUSTUNIT) pnum = CLUSTUNIT;
  blnum = *np;
  if(blnum > UNITINIT) blnum = UNITINIT;
  clusts = cbmalloc(pnum * sizeof(CLUSTER) + 1);
  clnum = 0;
  tails = cbmalloc(blnum * sizeof(CBMAP *) + 1);
  tpairs = cbmalloc(blnum * sizeof(ODPAIR) + 1);
  tnum = 0;
  for(i = 0; i < blnum; i++){
    if(i < pnum){
      if(!(mbuf = crget(scdb, (char *)&(pairs[i].id), sizeof(int), 0, -1, &msiz))) continue;
      clusts[clnum].ids = cblistopen();
      sprintf(numbuf, "%d:%d", pairs[i].id, pairs[i].score);
      cblistpush(clusts[clnum].ids, numbuf, -1);
      clusts[clnum].scores = cbmapload(mbuf, msiz);
      free(mbuf);
      setovec(clusts[clnum].scores, clusts[clnum].vector);
      clusts[clnum].point = pairs[i].score;
      clnum++;
    } else {
      if(!(mbuf = crget(scdb, (char *)&(pairs[i].id), sizeof(int), 0, -1, &msiz))) continue;
      tails[tnum] = cbmapload(mbuf, msiz);
      free(mbuf);
      tpairs[tnum].id = pairs[i].id;
      tpairs[tnum].score = pairs[i].score;
      tnum++;
    }
  }
  sumupclusters(clusts, &clnum);
  for(i = 0; i < tnum; i++){
    mi = -1;
    msc = 0.0;
    setovec(tails[i], ovec);
    for(j = 0; j < clnum; j++){
      settvec(tails[i], clusts[j].scores, tvec);
      sc = odvectorcosine(ovec, tvec, RELVECNUM);
      if(mi < 0 || sc > msc){
        mi = j;
        msc = sc;
      }
    }
    if(msc < 0.45) continue;
    sprintf(numbuf, "%d:%d", tpairs[i].id, tpairs[i].score);
    cblistpush(clusts[mi].ids, numbuf, -1);
  }
  for(i = 0; i < clnum && i < optclshow; i++){
    if(optnk){
      xmlprintf("<cluster code=\"%d\" hit=\"%d\"/>\n", i + 1, cblistnum(clusts[i].ids));
    } else {
      xmlprintf("<cluster code=\"%d\" hit=\"%d\">\n", i + 1, cblistnum(clusts[i].ids));
      cbmapiterinit(clusts[i].scores);
      for(j = 0; j < CLUSTKEYS && (kbuf = cbmapiternext(clusts[i].scores, NULL)) != NULL; j++){
        xmlprintf("<clkey score=\"%@\">%@</clkey>\n",
                  cbmapget(clusts[i].scores, kbuf, -1, NULL), kbuf);
      }
      xmlprintf("</cluster>\n");
    }
  }
  free(tpairs);
  for(i = 0; i < tnum; i++){
    cbmapclose(tails[i]);
  }
  free(tails);
  if(optclcode > 0 && optclcode - 1 < clnum){
    for(i = 0; i < cblistnum(clusts[optclcode-1].ids); i++){
      kbuf = cblistval(clusts[optclcode-1].ids, i, NULL);
      pairs[i].id = atoi(kbuf);
      kbuf = strchr(kbuf, ':') + 1;
      pairs[i].score = atoi(kbuf);
    }
    *np = i;
    odpairssort(pairs, *np);
  }
  for(i = 0; i < clnum; i++){
    cbmapclose(clusts[i].scores);
    cblistclose(clusts[i].ids);
  }
  free(clusts);
}


/* sum up clusters */
void sumupclusters(CLUSTER *clusts, int *np){
  SCWORD *scws;
  CBMAP *scores;
  const char *kbuf, *vbuf;
  char numbuf[ESTNUMBUFSIZ];
  int i, j, clnum, rcnt, hit, mi, si, ksiz, vsiz, pt, tvec[RELVECNUM];
  double msc, sc;
  clnum = *np;
  for(rcnt = 0; clnum > optclshow; rcnt++){
    hit = FALSE;
    for(i = 0; i < clnum && clnum > optclshow; i++){
      if(rcnt < 2 && cbmaprnum(clusts[i].scores) < RELVECNUM) continue;
      mi = -1;
      msc = 0.0;
      for(j = 0; j < clnum; j++){
        if(j == i) continue;
        settvec(clusts[i].scores, clusts[j].scores, tvec);
        sc = odvectorcosine(clusts[i].vector, tvec, RELVECNUM);
        if(mi < 0 || sc > msc){
          mi = j;
          msc = sc;
        }
      }
      if(rcnt < 1 && msc < 0.95) continue;
      if(rcnt < 2 && msc < 0.90) continue;
      if(rcnt < 3 && msc < 0.85) continue;
      if(rcnt < 4 && msc < 0.80) continue;
      if(rcnt < 5 && msc < 0.75) continue;
      if(clnum < optclshow * 2 && msc < 0.60) continue;
      if(msc < 0.35) continue;
      hit = TRUE;
      for(j = 0; j < cblistnum(clusts[mi].ids); j++){
        kbuf = cblistval(clusts[mi].ids, j, &ksiz);
        cblistpush(clusts[i].ids, kbuf, ksiz);
      }
      cbmapiterinit(clusts[mi].scores);
      while((kbuf = cbmapiternext(clusts[mi].scores, &ksiz)) != NULL){
        pt = 0;
        if((vbuf = cbmapget(clusts[i].scores, kbuf, ksiz, NULL)) != NULL) pt = atoi(vbuf);
        vsiz = sprintf(numbuf, "%d", pt + atoi(cbmapget(clusts[mi].scores, kbuf, ksiz, NULL)));
        cbmapput(clusts[i].scores, kbuf, ksiz, numbuf, vsiz, TRUE);
      }
      cbmapclose(clusts[mi].scores);
      cblistclose(clusts[mi].ids);
      clusts[i].point += clusts[mi].point;
      clnum--;
      if(mi != clnum) clusts[mi] = clusts[clnum];
      si = i == clnum ? mi : i;
      scws = cbmalloc(cbmaprnum(clusts[si].scores) * sizeof(SCWORD) + 1);
      cbmapiterinit(clusts[si].scores);
      for(j = 0; (kbuf = cbmapiternext(clusts[si].scores, &ksiz)) != NULL; j++){
        scws[j].kbuf = kbuf;
        scws[j].ksiz = ksiz;
        scws[j].point = atoi(cbmapget(clusts[si].scores, kbuf, ksiz, NULL));
      }
      cbqsort(scws, cbmaprnum(clusts[si].scores), sizeof(SCWORD), scwordcompare);
      scores = cbmapopen();
      for(j = 0; j < cbmaprnum(clusts[si].scores); j++){
        vsiz = sprintf(numbuf, "%d", scws[j].point);
        cbmapput(scores, scws[j].kbuf, scws[j].ksiz, numbuf, vsiz, TRUE);
      }
      cbmapclose(clusts[si].scores);
      clusts[si].scores = scores;
      setovec(clusts[si].scores, clusts[si].vector);
      free(scws);
      if(msc >= 0.85) i--;
    }
    if(!hit && rcnt >= 8) break;
  }
  cbqsort(clusts, clnum, sizeof(CLUSTER), clustercompare);
  *np = clnum;
}


/* compare two scores */
int scwordcompare(const void *a, const void *b){
  SCWORD *ap, *bp;
  ap = (SCWORD *)a;
  bp = (SCWORD *)b;
  return bp->point - ap->point;
}


/* compare two clusters */
int clustercompare(const void *a, const void *b){
  CLUSTER *ap, *bp;
  ap = (CLUSTER *)a;
  bp = (CLUSTER *)b;
  return bp->point - ap->point;
}


/* cut a uri to a directory */
char *cuturi(const char *uri, int level){
  char *buf, *pv;
  int i;
  buf = cbmemdup(uri, -1);
  for(i = 0; i < level; i++){
    if(!(pv = strrchr(buf, '/'))) break;
    *pv = '\0';
  }
  return buf;
}


/* show the preamble */
void showpreamble(void){
  xmlprintf("<?xml version=\"1.0\" encoding=\"%@\"?>\n", optocode ? optocode : ENCODING);
  if(optcss) xmlprintf("<?xml-stylesheet href=\"%@\" type=\"text/css\"?>\n", CSSFILE);
  if(optxsl) xmlprintf("<?xml-stylesheet href=\"%@\" type=\"application/xml\"?>\n", XSLFILE);
  if(optdtd) xmlprintf("<!DOCTYPE estxview SYSTEM \"%@\">\n", DTDFILE);
}


/* show a document */
void showdoc(const ODDOC *doc, int score, double rel, ESTWORD *words, int wnum){
  const CBLIST *nwords, *awords;
  CBMAP *scores;
  CBDATUM *ssel;
  const char *uri, *ruri, *title, *author, *rcpt, *mcast, *date, *type, *enc, *size, *word;
  char numbuf[ESTNUMBUFSIZ], *mbuf;
  int i, id, msiz, wsiz;
  id = oddocid(doc);
  uri = oddocuri(doc);
  ruri = oddocgetattr(doc, "realuri");
  title = oddocgetattr(doc, "title");
  author = oddocgetattr(doc, "author");
  rcpt = oddocgetattr(doc, "recipient");
  mcast = oddocgetattr(doc, "multicast");
  date = oddocgetattr(doc, "date");
  type = oddocgetattr(doc, "type");
  enc = oddocgetattr(doc, "encoding");
  size = oddocgetattr(doc, "size");
  if(rel >= 0.0){
    sprintf(numbuf, "%.2f", rel / 100);
    xmlprintf("<document id=\"%d\" similarity=\"%@\">\n", id, numbuf);
  } else if(score >= 0){
    xmlprintf("<document id=\"%d\" score=\"%d\">\n", id, score);
  } else {
    xmlprintf("<document id=\"%d\">\n", id);
  }
  xmlprintf("<uri>%@</uri>\n", uri);
  if(ruri && ruri[0] != '\0') xmlprintf("<realuri>%@</realuri>\n", ruri);
  if(title && title[0] != '\0') xmlprintf("<title>%@</title>\n", title);
  if(author && author[0] != '\0') xmlprintf("<author>%@</author>\n", author);
  if(rcpt && rcpt[0] != '\0') xmlprintf("<recipient>%@</recipient>\n", rcpt);
  if(mcast && mcast[0] != '\0') xmlprintf("<multicast>%@</multicast>\n", mcast);
  if(date && date[0] != '\0') xmlprintf("<date>%@</date>\n", date);
  if(type && type[0] != '\0') xmlprintf("<type>%@</type>\n", type);
  if(enc && enc[0] != '\0') xmlprintf("<encoding>%@</encoding>\n", enc);
  if(size && size[0] != '\0') xmlprintf("<size>%@</size>\n", size);
  if(!optnt){
    nwords = oddocnwords(doc);
    awords = oddocawords(doc);
    xmlprintf("<text>\n");
    for(i = 0; i < cblistnum(nwords); i++){
      xmlprintf("<word normal=\"%@\">%@</word>\n",
                cblistval(nwords, i, NULL), cblistval(awords, i, NULL));
    }
    xmlprintf("</text>\n");
  }
  if(optsnum > 0) showsummary(doc, words, wnum);
  if(!optnk && scdb && (mbuf = crget(scdb, (char *)&id, sizeof(int), 0, -1, &msiz)) != NULL){
    ssel = cbdatumopen("", 0);
    scores = cbmapload(mbuf, msiz);
    cbmapiterinit(scores);
    while((word = cbmapiternext(scores, &wsiz)) != NULL){
      cbdatumcat(ssel, word, wsiz);
      cbdatumcat(ssel, "\t", 1);
      cbdatumcat(ssel, cbmapget(scores, word, wsiz, NULL), -1);
      cbdatumcat(ssel, "\n", 1);
    }
    xmlprintf("<keywords>%?</keywords>\n", cbdatumptr(ssel));
    cbdatumclose(ssel);
    cbmapclose(scores);
    free(mbuf);
  }
  xmlprintf("</document>\n");
}


/* show summary of a document */
void showsummary(const ODDOC *doc, const ESTWORD *words, int wnum){
  const CBLIST *nwords, *awords;
  CBMAP *kmap, *tmap;
  const char *evword, *normal, *asis, *ibuf;
  int i, j, lnum, gshow, lshow, nwsiz, awsiz, cjk, tb, pv, bi;
  xmlprintf("<summary>");
  nwords = oddocnwords(doc);
  awords = oddocawords(doc);
  kmap = cbmapopen();
  tmap = cbmapopen();
  if(words){
    for(i = 0; i < wnum; i++){
      if(words[i].type == ESTCONDNOT) continue;
      if(words[i].evwords){
        for(j = 0; j < cblistnum(words[i].evwords); j++){
          evword = cblistval(words[i].evwords, j, NULL);
          cbmapput(kmap, evword, -1, (char *)&i, sizeof(int), TRUE);
          cbmapput(tmap, evword, -1, (char *)&i, sizeof(int), TRUE);
        }
      } else {
        cbmapput(kmap, words[i].word, -1, (char *)&i, sizeof(int), TRUE);
        cbmapput(tmap, words[i].word, -1, (char *)&i, sizeof(int), TRUE);
      }
    }
  }
  lnum = cblistnum(nwords);
  gshow = 0;
  /* show top words */
  lshow = 0;
  cjk = FALSE;
  for(i = 0; i < lnum && lshow < (words ? SUMTOP : optsnum); i++){
    normal = cblistval(nwords, i, &nwsiz);
    asis = cblistval(awords, i, &awsiz);
    if(awsiz < 1) continue;
    if(lshow > 0 && (!cjk || *(unsigned char *)asis < 0xe0)) xmlprintf(" ");
    if(nwsiz > 0 && (ibuf = cbmapget(kmap, normal, nwsiz, NULL)) != NULL){
      xmlprintf("<key term=\"%@\">%@</key>", normal, asis);
    } else {
      xmlprintf("%@", asis);
    }
    cjk = *(unsigned char *)asis >= 0xe0;
    gshow++;
    lshow++;
  }
  xmlprintf(" <delim>...</delim> ");
  /* show phrases around search words */
  tb = wnum > 0 && wnum < 4 ? (SUMWIDTH / 2) / wnum : 0;
  pv = i;
  while(i < lnum){
    if(cbmaprnum(kmap) > 0 && cbmaprnum(tmap) < 1){
      cbmapclose(tmap);
      tmap = cbmapdup(kmap);
    }
    normal = cblistval(nwords, i, &nwsiz);
    asis = cblistval(awords, i, &awsiz);
    if(awsiz > 0 && cbmapget(tmap, normal, nwsiz, NULL)){
      cbmapout(tmap, normal, nwsiz);
      bi = i - (SUMWIDTH + tb) / 2;
      bi = bi > pv ? bi : pv;
      lshow = 0;
      cjk = FALSE;
      for(j = bi; j < lnum && j <= bi + SUMWIDTH + tb; j++){
        normal = cblistval(nwords, j, &nwsiz);
        asis = cblistval(awords, j, &awsiz);
        if(awsiz < 1) continue;
        if(lshow > 0 && (!cjk || *(unsigned char *)asis < 0xe0)) xmlprintf(" ");
        if(nwsiz > 0 && (ibuf = cbmapget(kmap, normal, nwsiz, NULL)) != NULL){
          xmlprintf("<key term=\"%@\">%@</key>", normal, asis);
        } else {
          xmlprintf("%@", asis);
        }
        cjk = *(unsigned char *)asis >= 0xe0;
        gshow++;
        lshow++;
      }
      xmlprintf(" <delim>...</delim> ");
      i = j;
      pv = i;
    } else {
      i++;
    }
    if(gshow > optsnum) break;
  }
  /* infill the left space */
  if(pv < lnum - SUMWIDTH && optsnum - gshow > SUMWIDTH){
    lshow = 0;
    cjk = FALSE;
    for(i = pv; i < lnum && gshow <= optsnum; i++){
      asis = cblistval(awords, i, NULL);
      if(lshow > 0 && (!cjk || *(unsigned char *)asis < 0xe0)) xmlprintf(" ");
      xmlprintf("%@", asis);
      cjk = *(unsigned char *)asis >= 0xe0;
      gshow++;
      lshow++;
    }
    xmlprintf(" <delim>...</delim> ");
  }
  /* release resources */
  cbmapclose(tmap);
  cbmapclose(kmap);
  xmlprintf("</summary>\n");
}


/* show a document in tiny format */
void showdoctiny(int id, int score, double rel){
  char numbuf[ESTNUMBUFSIZ];
  if(rel >= 0.0){
    sprintf(numbuf, "%.2f", rel / 100);
    xmlprintf("<document id=\"%d\" similarity=\"%@\"/>\n", id, numbuf);
  } else if(score >= 0){
    xmlprintf("<document id=\"%d\" score=\"%d\"/>\n", id, score);
  } else {
    xmlprintf("<document id=\"%d\"/>\n", id);
  }
}


/* show a document specified by the ID */
int showdocbyid(int id){
  ODDOC *doc;
  int err, hit;
  err = FALSE;
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  xmlprintf("<query operator=\"id\">%d</query>\n", id);
  doc = NULL;
  hit = 0;
  if(opttiny){
    if(odcheck(odeum, id)){
      hit = 1;
    } else {
      id = -1;
      showerror("There is no corresponding document.", dpecode);
      err = TRUE;
    }
  } else {
    if((doc = odgetbyid(odeum, id)) != NULL){
      hit = 1;
    } else {
      showerror("There is no corresponding document.", dpecode);
      err = TRUE;
    }
  }
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n", hit, oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  if(opttiny){
    if(id != -1) showdoctiny(id, -1, -1.0);
  } else {
    if(doc){
      showdoc(doc, -1, -1.0, NULL, 0);
      oddocclose(doc);
    }
  }
  xmlprintf("</estxview>\n");
  return err ? FALSE : TRUE;
}


/* show a document specified by the URI */
int showdocbyuri(const char *uri){
  ODDOC *doc;
  int err, hit, id;
  err = FALSE;
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  xmlprintf("<query operator=\"uri\">%@</query>\n", uri);
  hit = 0;
  doc = NULL;
  id = -1;
  if(opttiny){
    if((id = odgetidbyuri(odeum, uri)) != -1){
      hit = 1;
    } else {
      showerror("There is no corresponding document.", dpecode);
      err = TRUE;
    }
  } else {
    if((doc = odget(odeum, uri)) != NULL){
      hit = 1;
    } else {
      showerror("There is no corresponding document.", dpecode);
      err = TRUE;
    }
  }
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n", hit, oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  if(opttiny){
    if(id != -1) showdoctiny(id, -1, -1.0);
  } else {
    if(doc){
      showdoc(doc, -1, -1.0, NULL, 0);
      oddocclose(doc);
    }
  }
  xmlprintf("</estxview>\n");
  return err ? FALSE : TRUE;
}


/* show documents related to a document of the ID */
int showdocbyrel(int id, const char *relsc){
  ODDOC *doc, *tdoc;
  CBMAP *scores, *tsc, *dirs;
  CBLIST *twords;
  CBDATUM *phrase;
  ESTWORD *words;
  ODPAIR *pairs;
  const char *word, *score;
  char *mbuf, *tmbuf, *tmp;
  int i, err, msiz, tmsiz, wsiz, wnum, pnum, lnum, show;
  int ovec[RELVECNUM], tvec[RELVECNUM];
  err = FALSE;
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  if(relsc){
    xmlprintf("<query operator=\"relsc\">%?</query>\n", relsc);
  } else {
    xmlprintf("<query operator=\"related\">%d</query>\n", id);
  }
  xmlprintf("<option max=\"%d\" tfidf=\"%@\" clshow=\"%d\" clcode=\"%d\""
            " drep=\"%d\" expr=\"none\" sort=\"none\"/>\n",
            optmax, optni ? "false" : "true", optclshow, optclcode, optdrep);
  doc = NULL;
  mbuf = NULL;
  scores = NULL;
  twords = NULL;
  phrase = NULL;
  words = NULL;
  pairs = NULL;
  pnum = 0;
  if(!relsc && !(doc = odgetbyid(odeum, id))){
    showerror("There is no seed document.", dpecode);
    err = TRUE;
  } else if(!relsc && !(mbuf = crget(scdb, (char *)&id, sizeof(int), 0, -1, &msiz))){
    showerror("The document has no score.", -1);
    err = TRUE;
  } else {
    if(relsc){
      scores = cbmapopen();
      twords = cbsplit(relsc, -1, "\n");
      for(i = 0; i < cblistnum(twords); i++){
        word = cblistval(twords, i, NULL);
        if(!(score = strchr(word, '\t'))) continue;
        cbmapput(scores, word, score - word, score + 1, -1, TRUE);
      }
      cblistclose(twords);
      twords = cbmapkeys(scores);
    } else {
      scores = cbmapload(mbuf, msiz);
      twords = cbmapkeys(scores);
    }
    phrase = cbdatumopen("", 0);
    for(i = 0; i < RELKEYS && i < cblistnum(twords); i++){
      word = cblistval(twords, i, &wsiz);
      if(i > 0) cbdatumcat(phrase, " [OR] ", -1);
      cbdatumcat(phrase, word, wsiz);
    }
    words = estsearchwords(cbdatumptr(phrase), &wnum, FALSE);
    pairs = estsearch(odeum, words, wnum, UNITINIT, !optni, &pnum, &lnum, FALSE, FALSE, 0);
    setovec(scores, ovec);
    for(i = 0; i < pnum; i++){
      if((tmbuf = crget(scdb, (char *)&(pairs[i].id), sizeof(int), 0, -1, &tmsiz)) != NULL){
        tsc = cbmapload(tmbuf, tmsiz);
        free(tmbuf);
      } else {
        tsc = cbmapopen();
      }
      settvec(scores, tsc, tvec);
      cbmapclose(tsc);
      pairs[i].score = odvectorcosine(ovec, tvec, RELVECNUM) * 10000;
      if(pairs[i].score >= 9999) pairs[i].score = 10000;
    }
    odpairssort(pairs, pnum);
    for(i = 0; i < pnum; i++){
      if(pairs[i].score < 1){
        pnum = i;
        break;
      }
    }
  }
  if(pnum < 1 && !err){
    showerror("There is no corresponding document.", DP_ENOITEM);
    err = TRUE;
  }
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n", pnum, oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  if(pairs){
    if(scdb && optclshow > 0 && pnum > 0) showclusters(pairs, &pnum);
    show = 0;
    dirs = NULL;
    if(optdrep > 0) dirs = cbmapopen();
    for(i = 0; i < pnum && show < optmax; i++){
      if(opttiny && !dirs){
        if(!odcheck(odeum, pairs[i].id)) continue;
        showdoctiny(pairs[i].id, -1, pairs[i].score);
        show++;
      } else {
        if(!(tdoc = odgetbyid(odeum, pairs[i].id))) continue;
        if(dirs){
          tmp = cuturi(oddocuri(tdoc), optdrep);
          if(cbmapget(dirs, tmp, -1, NULL)){
            free(tmp);
            oddocclose(tdoc);
            continue;
          }
          cbmapput(dirs, tmp, -1, "", 0, FALSE);
          free(tmp);
        }
        if(opttiny){
          showdoctiny(pairs[i].id, -1, pairs[i].score);
        } else {
          showdoc(tdoc, -1, pairs[i].score, NULL, 0);
        }
        oddocclose(tdoc);
        show++;
      }
    }
    if(dirs) cbmapclose(dirs);
  }
  if(pairs) free(pairs);
  if(words) estfreewords(words, wnum);
  if(phrase) cbdatumclose(phrase);
  if(twords) cblistclose(twords);
  if(scores) cbmapclose(scores);
  if(mbuf) free(mbuf);
  if(doc) oddocclose(doc);
  xmlprintf("</estxview>\n");
  return err ? FALSE : TRUE;
}


/* show documents by searching with words */
int showdocbywords(ESTWORD *words, int wnum){
  ODDOC *doc;
  ODPAIR *pairs, pswap;
  CBMAP *dirs;
  const char *dstr;
  char *tmp;
  int i, err, unit, ldnum, pnum, lnum, date, show;
  err = FALSE;
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  pairs = NULL;
  unit = UNITINIT;
  if(optdrep > 0) unit *= optdrep + 1;
  if(strcmp(optstype, "score")) unit = INT_MAX;
  do {
    if(pairs){
      free(pairs);
      unit *= UNITINC;
    }
    pairs = estsearch(odeum, words, wnum, unit, !optni, &pnum, &lnum,
                      !strcmp(optetype, "regex"), !strcmp(optetype, "wild"), REEVMAX);
  } while(pnum < optmax * RESROOM && wnum > 1 && lnum > 0);
  if(!strcmp(optstype, "date") || !strcmp(optstype, "r-date")){
    for(i = 0; i < pnum; i++){
      date = 0;
      if(!dtdb){
        if((doc = odgetbyid(odeum, pairs[i].id)) != NULL){
          if((dstr =  oddocgetattr(doc, "date")) != NULL) date = eststrmktime(dstr);
          oddocclose(doc);
        }
      } else {
        dpgetwb(dtdb, (char *)&(pairs[i].id), sizeof(int), 0, sizeof(int), (char *)&date);
      }
      pairs[i].score = date;
    }
    odpairssort(pairs, pnum);
  }
  if(cbstrfwmatch(optstype, "r-")){
    for(i = 0; i < pnum / 2; i++){
      pswap = pairs[i];
      pairs[i] = pairs[pnum-i-1];
      pairs[pnum-i-1] = pswap;
    }
  }
  ldnum = 0;
  for(i = 0; i < wnum; i++){
    ldnum = words[i].dnum;
    if(i < 1){
      xmlprintf("<query hit=\"%d\">%@</query>\n", words[i].dnum, words[i].word);
    } else {
      xmlprintf("<query hit=\"%d\" operator=\"%@\">%@</query>\n", words[i].dnum,
                words[i].type == ESTCONDAND ? "and" : words[i].type == ESTCONDOR ? "or" : "not",
                words[i].word);
    }
  }
  xmlprintf("<option max=\"%d\" tfidf=\"%@\" clshow=\"%d\" clcode=\"%d\""
            " drep=\"%d\" expr=\"%?\" sort=\"%?\"/>\n",
            optmax, optni ? "false" : "true", optclshow, optclcode, optdrep, optetype, optstype);
  if(pnum < 1){
    showerror("There is no corresponding document.", dpecode);
    err = TRUE;
  }
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n",
            wnum < 2 ? ldnum : pnum, oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  if(scdb && optclshow > 0 && pnum > 0) showclusters(pairs, &pnum);
  show = 0;
  dirs = NULL;
  if(optdrep > 0) dirs = cbmapopen();
  for(i = 0; i < pnum && show < optmax; i++){
    if(opttiny && !dirs){
      if(!odcheck(odeum, pairs[i].id)) continue;
      showdoctiny(pairs[i].id, pairs[i].score, -1.0);
      show++;
    } else {
      if(!(doc = odgetbyid(odeum, pairs[i].id))) continue;
      if(dirs){
        tmp = cuturi(oddocuri(doc), optdrep);
        if(cbmapget(dirs, tmp, -1, NULL)){
          free(tmp);
          oddocclose(doc);
          continue;
        }
        cbmapput(dirs, tmp, -1, "", 0, FALSE);
        free(tmp);
      }
      if(opttiny){
        showdoctiny(pairs[i].id, pairs[i].score, -1.0);
      } else {
        showdoc(doc, pairs[i].score, -1.0, words, wnum);
      }
      oddocclose(doc);
      show++;
    }
  }
  if(dirs) cbmapclose(dirs);
  free(pairs);
  xmlprintf("</estxview>\n");
  return err ? FALSE : TRUE;
}


/* output information of all registered documents */
int dodlist(const char *name){
  CURIA *docsdb;
  ODDOC *doc;
  char path[ESTPATHBUFSIZ], *kbuf;
  int ksiz, id;
  if(!(odeum = odopen(name, OD_OREADER))){
    fprintf(stderr, "%s: %s: could not open: %s\n", progname, name, dperrmsg(dpecode));
    return 1;
  }
  sprintf(path, "%s%c%s", name, ESTPATHCHR, ESTSCDBNAME);
  scdb = cropen(path, CR_OREADER, -1, -1);
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  xmlprintf("<query operator=\"dlist\"/>\n");
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n",
            oddnum(odeum), oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  docsdb = odidbdocs(odeum);
  criterinit(docsdb);
  while((kbuf = criternext(docsdb, &ksiz)) != NULL){
    if(ksiz == sizeof(int) && (id = *(int *)kbuf) > 0){
      if(opttiny){
        showdoctiny(id, -1, -1.0);
      } else if((doc = odgetbyid(odeum, id)) != NULL){
        showdoc(doc, -1, -1.0, NULL, 0);
        oddocclose(doc);
      }
    }
    free(kbuf);
  }
  xmlprintf("</estxview>\n");
  if(scdb) crclose(scdb);
  odclose(odeum);
  return 0;
}


/* output information of all index terms */
int dowlist(const char *name){
  CURIA *indexdb;
  char *kbuf;
  int ksiz;
  if(!(odeum = odopen(name, OD_OREADER))){
    fprintf(stderr, "%s: %s: could not open: %s\n", progname, name, dperrmsg(dpecode));
    return 1;
  }
  showpreamble();
  xmlprintf("<estxview version=\"%@\" options=\"%@\">\n", _EST_VERSION, getoptionstr());
  xmlprintf("<meta>\n");
  xmlprintf("<query operator=\"wlist\"/>\n");
  xmlprintf("<total hit=\"%d\" dnum=\"%d\" wnum=\"%d\"/>\n",
            oddnum(odeum), oddnum(odeum), odwnum(odeum));
  xmlprintf("</meta>\n");
  indexdb = odidbindex(odeum);
  criterinit(indexdb);
  xmlprintf("<terms>\n");
  while((kbuf = criternext(indexdb, &ksiz)) != NULL){
    xmlprintf("<term hit=\"%d\">%@</term>\n",
              crvsiz(indexdb, kbuf, ksiz) / sizeof(ODPAIR), kbuf);
    free(kbuf);
  }
  xmlprintf("</terms>\n");
  xmlprintf("</estxview>\n");
  odclose(odeum);
  return 0;
}



/* END OF FILE */
