/*************************************************************************************************
 * Command line utility for system integration
 *                                                      Copyright (C) 2003-2006 Mikio Hirabayashi
 * This file is part of Estraier, a personal full-text search system.
 * Estraier is free software; you can redistribute it and/or modify it under the terms of the GNU
 * General Public License as published by the Free Software Foundation; either version 2 of the
 * License, or any later version.
 * Estraier is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY;
 * without even the implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU General Public License for more details.
 * You should have received a copy of the GNU General Public License along with Estraier;
 * if not, write to the Free Software Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA.
 *************************************************************************************************/


#include "estcommon.h"


/* global variables */
const char *progname = NULL;             /* program name */


/* function prototypes */
int main(int argc, char **argv);
void usage(void);
int runcgiparam(int argc, char **argv);
int runhtmlesc(int argc, char **argv);
int runurlenc(int argc, char **argv);


/* main routine */
int main(int argc, char **argv){
  int rv;
  estputenv("LANG", ESTLOCALE);
  estputenv("LC_ALL", ESTLOCALE);
  cbstdiobin();
  progname = argv[0];
  if(argc < 2) usage();
  rv = 0;
  if(!strcmp(argv[1], "prefix")){
    printf("%s\n", _EST_PREFIX);
    rv = 0;
  } else if(!strcmp(argv[1], "bindir")){
    printf("%s\n", _EST_BINDIR);
    rv = 0;
  } else if(!strcmp(argv[1], "libexecdir")){
    printf("%s\n", _EST_LEXEDIR);
    rv = 0;
  } else if(!strcmp(argv[1], "datadir")){
    printf("%s\n", _EST_DATADIR);
    rv = 0;
  } else if(!strcmp(argv[1], "version")){
    printf("%s\n", _EST_VERSION);
    rv = 0;
  } else if(!strcmp(argv[1], "cgiparam")){
    rv = runcgiparam(argc, argv);
  } else if(!strcmp(argv[1], "htmlesc")){
    rv = runhtmlesc(argc, argv);
  } else if(!strcmp(argv[1], "urlenc")){
    rv = runurlenc(argc, argv);
  } else {
    usage();
  }
  return rv;
}


/* print the usage and exit */
void usage(void){
  fprintf(stderr, "%s: utility for system integration\n", progname);
  fprintf(stderr, "\n");
  fprintf(stderr, "usage:\n");
  fprintf(stderr, "  %s prefix\n", progname);
  fprintf(stderr, "  %s bindir\n", progname);
  fprintf(stderr, "  %s libexecdir\n", progname);
  fprintf(stderr, "  %s datadir\n", progname);
  fprintf(stderr, "  %s version\n", progname);
  fprintf(stderr, "  %s cgiparam name [qstr]\n", progname);
  fprintf(stderr, "  %s htmlesc str\n", progname);
  fprintf(stderr, "  %s urlenc str\n", progname);
  fprintf(stderr, "\n");
  exit(1);
}


/* extract parameters of CGI */
int runcgiparam(int argc, char **argv){
  CBDATUM *datum;
  CBMAP *params;
  CBLIST *pairs;
  const char *name, *qstr, *rkey;
  char *tmp, cc, *key, *val, *dkey, *dval;
  int i, ic, err, rksiz;
  name = NULL;
  qstr = NULL;
  for(i = 2; i < argc; i++){
    if(!name){
      name = argv[i];
    } else if(!qstr){
      qstr = argv[i];
    } else {
      usage();
    }
  }
  if(!name) usage();
  if(qstr){
    tmp = cbmemdup(qstr, -1);
  } else {
    datum = cbdatumopen("", 0);
    while((ic = getchar()) != EOF){
      cc = (char)ic;
      cbdatumcat(datum, &cc, 1);
    }
    tmp = cbdatumtomalloc(datum, NULL);
  }
  params = cbmapopen();
  pairs = cbsplit(tmp, -1, "&");
  err = FALSE;
  for(i = 0; i < cblistnum(pairs); i++){
    key = cbmemdup(cblistval(pairs, i, NULL), -1);
    if((val = strchr(key, '=')) != NULL){
      *(val++) = '\0';
      dkey = cburldecode(key, NULL);
      dval = cburldecode(val, NULL);
      cbmapput(params, dkey, -1, dval, -1, FALSE);
      free(dval);
      free(dkey);
    }
    free(key);
  }
  cblistclose(pairs);
  if(name[0] != '\0'){
    if((rkey = cbmapget(params, name, -1, &rksiz)) != NULL){
      for(i = 0; i < rksiz; i++){
        putchar(rkey[i]);
      }
    } else {
      fprintf(stderr, "%s: %s: no such parameter\n", progname, name);
      err = TRUE;
    }
  } else {
    cbmapiterinit(params);
    while((rkey = cbmapiternext(params, NULL)) != NULL){
      printf("%s\t%s\n", rkey, cbmapget(params, rkey, -1, NULL));
    }
  }
  cbmapclose(params);
  free(tmp);
  return err ? 1 : 0;
}


/* escape a string for HTML */
int runhtmlesc(int argc, char **argv){
  char *esc;
  if(argc != 3) return 1;
  esc = cbxmlescape(argv[2]);
  printf("%s", esc);
  free(esc);
  return 0;
}


/* encode a string for URL */
int runurlenc(int argc, char **argv){
  char *esc;
  if(argc != 3) return 1;
  esc = cburlencode(argv[2], -1);
  printf("%s", esc);
  free(esc);
  return 0;
}



/* END OF FILE */
